(* ByuSS.isl *)
(*
Copyright (c) 1994, 1995 Xerox Corporation.  All Rights Reserved.
"Use, reproduction and preparation of derivative works of this
software is permitted, but only for non-comercial research purposes.
Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.
This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES."
*)
(* $Id: ByuSS.isl,v 1.18 1995/09/01 14:38:04 spreitze Exp $ *)
(* Last edited by Mike Spreitzer September 1, 1995 7:37 am PDT *)

(* 
Specifies (1) the Bayou server-server protocol used for anti-entropy,
and (2) how new secondaries are created.  A Bayou server has all three
of the object types defined in this interface: Src, Snk, and Mgmt.
The first two, Src and Snk, are for the two roles in one-directional
Anti-Entropy.  These two roles are clearly distinguished from each
other, and from other server functionality, in the hope that this will
facilitate doing Anti-Entopy over non-interactive media, such as
floppy disks and PCMCIA memory cards.  The third object type, Mgmt,
adds everything not already covered by the client/server interface and
the two Anti-Entropy roles.  At present this amounts to creation of
new secondary servers.

The comments for Anti-Entropy are in terms of the following pointers
into the Write Log:

o:	last omitted Write;
s:	last Write reflected in the Tuple Store checkpoint on disk;
c:	last comitted Write;
t:	last Write (we're changing this name to "f").

*)

INTERFACE ByuSS IMPORTS ByuTypes END;

TYPE String = ByuTypes.String;
TYPE AuthChallenge = ByuTypes.AuthChallenge;
TYPE AuthResponse = ByuTypes.AuthResponse;

TYPE TID = String;

EXCEPTION BadTid;

EXCEPTION Down;
  (* Raised while server cannot fulfill its contract.
     Implies BadTid for procedures that can raise both. *)

TYPE ObjectHandle = RECORD sbh: String, mstid: String END;

EXCEPTION AccessDenied;  
  (* Raised if caller hasn't proved it has a needed privilege. *)

TYPE Src = OBJECT
    METHODS
    
    DoAE(to: ObjectHandle, tid: TID,
         cCS: ByuTypes.CStamp, tVV: ByuTypes.Vector,
         s1p: ByuTypes.Principal, C1: AuthChallenge,
         toe: ByuTypes.Bytes, s1certs: ByuTypes.CertSet)
        RAISES Down, AccessDenied END
    (* A server (or any other Snk) S1 calls this procedure to request
       that server (or any other Src) S2 do one-directional Anti-Entropy
       to update S1.  For this purpose, S1 allocates a
       transaction ID /tid/, calls S2.DoAE(S1, tid, S1.cCS, S1.tVV, ..),
       and terminates that transaction when the call returns (if not
       sooner due to an exception).  A transaction may time out if S2
       isn't lively enough; BadTid is raised for calls on S1 when that
       happens; S1 could then start over again with a larger timeout
       for the transaction.  Timeout limits the time between return from
       one call on S1 and start of next call.  /toe/ is the signature
       of s1p for cat(to.sbh, ";", to.mstid).  S1 is passed as an
       ObjectHandle, instead of directly as a Snk, to give the Bayou code
       in S2 explicit control over when S1 is imported (we think this
       became less interesting with version 1.8 of ILU).
       *)
    
    END;
(* A passive "source" of one-directional Anti-Entropy. *)

TYPE StartResult = RECORD
    sr-tid: TID,
    sr-cCS: ByuTypes.CStamp,
    sr-tVV: ByuTypes.Vector,
    sr-s1p: ByuTypes.Principal,	(* user running S1 *)
    sr-s1ide: ByuTypes.Bytes,	(* sign. of s1p for ByuSid_unparse(S1) *)
    sr-R2: AuthResponse,	(* wrt C2, s1p *)
    sr-C1: AuthChallenge,	(* challenge for caller *)
    sr-s1certs: ByuTypes.CertSet	(* about S1 *)
    END;

TYPE DbFragment = SEQUENCE OF TableFragment;

TYPE TableFragment = RECORD
    fragTableName: String,
    fragTups: ByuTypes.Relation
    END;

TYPE CmtNews = SHORT SEQUENCE OF CmtNewsItem;
TYPE CmtNewsItem = UNION WidInfo, WrInfo END;

TYPE WriteSeq = SHORT SEQUENCE OF ByuTypes.IdedWrite;

TYPE WrInfo = RECORD
    wiId: ByuTypes.WriteID,
    wiWr: ByuTypes.Write,
    wiAc: BOOLEAN
    END;

TYPE WidInfo = RECORD
    wiId: ByuTypes.WriteID,
    wiAc: BOOLEAN
    END;

TYPE Duration = INTEGER;  (* length of time in msecs *)

EXCEPTION ProtoErr: String;
  (* Raised when caller violates the AE protocol. *)

TYPE Snk = OBJECT
    METHODS
    
    PullFrom(tid: TID, s2p: ByuTypes.Principal,
             R1: AuthResponse, C2: AuthChallenge,
             s2certs: ByuTypes.CertSet): AuthResponse
	RAISES Down, AccessDenied, BadTid, ProtoErr END,
    (* When S1 calls S2.DoAE(S1, ...), S2 first calls
       S1.PullFrom to submit S2's credentials.  R1 is
       wrt C1 & s2p; C2 is a challenge for S1.  Result should be
       response to C2 wrt s1p.  S2 checks result to authenticate S1. *)
    
    ReceivePush(timeout: CARDINAL, s2p: ByuTypes.Principal,
                C2: AuthChallenge, s2certs: ByuTypes.CertSet)
	: StartResult
	RAISES Down, AccessDenied END,
    (* Alternatively to DoAE/PullFrom, S2 can, without exporting any
       network interfaces at all, initiate a push to S1 by calling
       S1.ReceivePush(timeout, ...).  The timeout is expressed in
       seconds.  If it turns out to be too short, S2 can
       start over again with a larger timeout. *)
    
    AuthPush(tid: TID, R1: AuthResponse)
	RAISES Down, AccessDenied, BadTid, ProtoErr END,
    (* After calling ReceivePush, S2 calls S1.AuthPush to submit
       S2's response to C1 wrt s2p.
       After successful return, S2 can make other calls with tid. *)
    
    StartJump(tid: TID, oVV: ByuTypes.Vector, oCS: ByuTypes.CStamp,
	oWID: ByuTypes.WriteID, sVV: ByuTypes.Vector,
	sCS: ByuTypes.CStamp, sWID: ByuTypes.WriteID,
	frag: DbFragment)
	RAISES Down, BadTid, ProtoErr END,
    (* When S1.c < S2.o (i.e., S2 has discarded Writes
       that S1 doesn't yet know to be committed --- if at all),
       S2 "jumps" S1 forward by passing S2's TupleStore contents
       (at point s) and the Writes in S2's log in the range (o, s],
       where s may be anywhere in the range [S2.s, S2.c],
       and o may be anywhere in the range [S2.o, s].
       S2 begins this process by calling S1.StartJump.
       S2 passes its TupleStore contents as a series of fragments,
       one of which may be passed as an argument to StartJump.
       Then LearnCommitted and ContinueJump are called as necessary
       to send the needed Writes and database contents.
       Finally FinishJump exactly once; this sequence of calls is
       aborted if and when BadTid is raised.
       [We wouldn't need to send Writes in (o, s] between StartJump and
       FinishJump if we could later send those Writes in reverse
       order (along with the necessary deltas to the pointer
       characteristics) --- letting us profit even if transmission of
       those Writes is interrupted.]
       *)
    
    ContinueJump(tid: TID, frag: DbFragment)
	RAISES Down, BadTid, ProtoErr END,
    
    FinishJump  (tid: TID, frag: DbFragment)
	RAISES Down, BadTid, ProtoErr END,
    
    LearnCommitted(tid: TID, startAt: ByuTypes.CStamp, news: CmtNews)
	RAISES Down, BadTid, ProtoErr END,
    (* S2 uses this to inform S1 of committed writes. *)
    
    LearnTentative(tid: TID, writes: WriteSeq)
	RAISES Down, BadTid, ProtoErr END,
    (* S2 uses this to inform S1 of tentative writes. *)
    
    FinishPush(tid: TID, pAge: Duration, pID: ByuTypes.ServerID)
	RAISES Down, ProtoErr END
    (* Last step; transmits primary ID and age. *)
    
    END;
(* The "sink" side of one-directional Anti-Entropy. *)

TYPE CreateMidstep = RECORD
    cm-tid: TID,
    cm-sp: ByuTypes.Principal,	(* user running existing server *)
    cm-spIDe: ByuTypes.Bytes,	(* sign. wrt sp, ByuSid_unparse(svr id) *)
    cm-Rc: AuthResponse,	(* response to Cc wrt sp *)
    cm-Cs: AuthChallenge,	(* challenge for caller *)
    cm-scerts: ByuTypes.CertSet (* about server *)
    END;

TYPE Mgmt = OBJECT
    METHODS
    
    StartCreatingSecondary(
	cp: ByuTypes.Principal,	(* user running caller *)
	Cc: AuthChallenge,	(* challenge for server *)
	ccerts: ByuTypes.CertSet,	(* about caller *)
	timeout: CARDINAL)
	: CreateMidstep RAISES Down, AccessDenied END,
    (* Creating a new server starts with an authentication exchange.
       The second step, FinishCreatingSecondary, must be called
       within the given timeout of StartCreatingSecondary. *)
    
    FinishCreatingSecondary(tid: TID, Rs: AuthResponse): ByuTypes.ServerID
	RAISES Down, AccessDenied END
    
    END;
(* Each existing server has this type, to facilitate creation
   of new secondaries.  We currently use the filesystem to do most of
   the initialization of the new secondary, and contact an existing
   server only to get the ID for the new server.  We plan to eliminate
   filesystem usage, and mediate creation of secondaries entirely
   through the network interface. *)
