(* ByuCS.isl *)
(*
Copyright (c) 1994, 1995 Xerox Corporation.  All Rights Reserved.
"Use, reproduction and preparation of derivative works of this
software is permitted, but only for non-comercial research purposes.
Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.
This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES."
*)
(* Doug Terry, June 28, 1994 3:04 pm PDT *)
(* Last tweaked by Mike Spreitzer August 31, 1995 8:51 am PDT *)
(* $Id: ByuCS.isl,v 1.12 1995/08/31 15:52:42 spreitze Exp $ *)

(* 
   Specifies the Bayou client-server protocol used for performing Reads
   and Writes on a Bayou database. 
 *)

INTERFACE ByuCS BRAND Version1 IMPORTS ByuTypes END;

TYPE String = ByuTypes.String;
TYPE AuthChallenge = ByuTypes.AuthChallenge;
TYPE AuthResponse = ByuTypes.AuthResponse;

TYPE Duration = INTEGER;  (* length of time in msecs *)

TYPE ReadResult = OPTIONAL ReadSuccess;
TYPE ReadSuccess = RECORD  (* results returned from a successful Read *)
    results: ByuTypes.Relation,
    relevant: ByuTypes.VectorPair
    END;

TYPE Update = ByuTypes.NormalUpdate;
TYPE UpdateOp = ByuTypes.NormalUpdateOp;
TYPE UpdateSet = SHORT SEQUENCE OF Update;

TYPE WriteResult = OPTIONAL ByuTypes.WriteID;

TYPE WriteStatus = ENUMERATION Committed, Tentative, Unknown END;  
  (* status of a Write *)

TYPE AccessDeniedReason = ENUMERATION
	BadId,		(* SessionID not currently valid *)
	LackPrivilege	(* client doesn't have needed privilege *)
	END;

EXCEPTION NotCurrent: ByuTypes.VectorPair;  
  (* raised if server is not sufficiently up-to-date *)
EXCEPTION AccessDenied: AccessDeniedReason;  
  (* raised if caller hasn't proved it has a needed privilege *)
EXCEPTION MalformedQuery: String;
  (* obvious *)
EXCEPTION Down;
  (* raised while server cannot fulfill its contract *)

TYPE SessionID = String;

TYPE BeginSessionResult = RECORD
	bsrU: ByuTypes.Principal, (* user running this server now *)
	bsrSid: ByuTypes.Bytes,	(* sign. of ByuSid_unparse() wrt bsrU *)
	bsrId: SessionID,	(* identifies the session created *)
	bsrTtl: Duration,	(* granted time to live *)
	bsrRc: AuthResponse,	(* wrt Cc, bsrU *)
	bsrCs: AuthChallenge,	(* fresh challenge for the caller *)
	bsrCerts: ByuTypes.CertSet (* show bsrU has Server privilege *)
	END;

TYPE Db = OBJECT
    METHODS

        BeginSession (client: ByuTypes.Principal, Cc: AuthChallenge,
                      certs: ByuTypes.CertSet, ttl: Duration)
                    : BeginSessionResult RAISES Down, AccessDenied END,
           (* To start a new client/server session, a client calls
              this procedure, passing: the public key (= identity) of
              the user running the client, a fresh challenge Cc for
              the server, a certificate set establishing the client
              user's database privileges, and a requested time-to-live
              for the session.  The time-to-live limits the time between
              the return from one call on server and the start of the
              next call.  The server replies with: the public
              key (= identity) of the user running the server, the
              signature of bsrU on ByuSid_unparse(the server's ID)
              (i.e., MD-5 hash encrypted with RSA & private key),
              a fresh identifier for the new session, the granted initial
              time-to-live (which may be larger or smaller than that
              requested), Cc signed by the server user, a fresh
              challenge for the client to sign, and a certificate set
              establishing the server user's Server privilege for the
              database.  The server ID is signed to make man-in-the-middle
              attacks harder.  May raise AccessDenied if the
              server grants no database privileges to the client user.
              If no exceptions raised, the client authenticates the
              server user by checking bsrSid against bsrU and the
              expected server ID, and checking bsrRc against bsrU and
              Cc, then looks at bsrCerts and his own revocation cache
              to decide whether the server user has Server privilege.
              If so, the client is willing
              to use the session.  The client then calls
              AuthSessionClient, passing: the session ID returned from
              BeginSession, and the response to bsrCs using the
              client's private key; the BOOLEAN result indicates
              whether the server thinks the client passed this
              authentication test.  If FALSE, the session is over; if
              TRUE, the client then proceeds to make other calls in
              the session.  All security details are temporary (and
              known to be not great); the real deal will come when ILU
              starts providing authentication, integrity, and secrecy. *)

        AuthSessionClient (si: SessionID, Rs: AuthResponse): BOOLEAN
          RAISES Down, AccessDenied END,
            (* see above *)
        
        RefreshSession (si: SessionID, ttl: Duration): Duration
          RAISES Down, AccessDenied END,
            (* If a client wants to use a session beyond its current
               time-to-live, the client refreshes the session before
               it times out.  Returns the new time-to-live, which
               may be larger or smaller than requested. *)

        RevExch (si: SessionID, revs: ByuTypes.CertSet): ByuTypes.CertSet
          RAISES Down, AccessDenied END,
            (* Ensures the server is aware of the given revocations,
               and fetches from the server revocations of the Server
               privilege.
               Requires no privileges. *)

        ASYNCHRONOUS EndSession (si: SessionID),
            (* A polite client calls this when done with a session,
               to expedite the server's reclamation of related
               resources. *)

        GetVector (si: SessionID): ByuTypes.VectorPair
          RAISES Down, AccessDenied END,
            (* Returns the contacted server's current vector pair
              indicating what writes and commits have been seen by
              the server.  Requires Read privilege. *)

        InformClientNeeds (si: SessionID, minS: ByuTypes.VectorPair,
                           age: Duration, maxWait: Duration): Duration 
          RAISES Down, NotCurrent, AccessDenied END,
            (* Informs the server that the client is expected to perform
              Read operations in the near future with the given "minS"
              and "age" values.  Result is length of time over which
              the server will try to meet the age criterion.
              Requires Read privilege. *)

        ASYNCHRONOUS RetractClientNeeds (si: SessionID),
            (* Tells the server that the given client no longer needs
               it to maintain a previously specified level of currency. *)

        Read (si: SessionID, query: ByuTypes.Query,
              committedOnly: BOOLEAN, minS: ByuTypes.VectorPair,
              age: Duration): ReadResult 
          RAISES Down, NotCurrent, AccessDenied, MalformedQuery END,
            (* Performs the query against the server's database provided
              its state dominates the "minS" and that its
              database contains all writes that were committed more than
              "age" time units ago.  Requires Read privilege. *)

        Write (si: SessionID, deps: ByuTypes.DependencySet,
               updates: UpdateSet, merge: ByuTypes.MergeProc,
               minS: ByuTypes.VectorPair): WriteResult
          RAISES Down, NotCurrent, AccessDenied END,
            (* Submits the Write {deps, updates, merge, st's user,
               st's user certificates} to the server,
               provided the minS restriction is met.  Requires
               Write privilege. *)

        GetWriteStatus (si: SessionID, wid: ByuTypes.WriteID)
          : WriteStatus RAISES Down, AccessDenied END
            (* Returns whether the write has been committed, is
              tentative (in the opinion of the contacted server), or
              hasn't been seen by the server.
              Requires Read privilege. *)

    END;
