/* ByuAPI.h */
/*
Copyright (c) 1994, 1995 Xerox Corporation.  All Rights Reserved.
"Use, reproduction and preparation of derivative works of this
software is permitted, but only for non-comercial research purposes.
Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.
This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES."
*/
/* $Id: ByuAPI.h,v 1.6 1995/08/15 18:16:26 spreitze Exp $ */
/* Last edited by Mike Spreitzer August 15, 1995 11:16 am PDT */

#ifndef _BYUAPI_H
#define _BYUAPI_H

#include <assert.h>

/********************************/
/*        General Issues        */
/********************************/

/* Allocated storage appearing in results is "malloc"ed;
   it is the caller's responsibility to free that storage. 
   This includes error strings returned by various calls
*/

/********************************/
/*    Relational Data Basics    */
/********************************/

typedef int ByuAPI_Bool;

typedef char *ByuAPI_RelationName;

typedef enum {			/* Basic data types available */
  ByuAPI_bool, ByuAPI_int, ByuAPI_card, ByuAPI_real, ByuAPI_string
} ByuAPI_Datatype;

typedef struct {		/* Definition of a fieldname in a
				 * tuple */
  char             *fd_name;
  ByuAPI_Datatype  fd_type;
  int fd_bytelen;	/* Space required for field in bytes */
  /* This field is not being set currently */
} ByuAPI_FieldDefn;

typedef struct {               /* Field descriptions for a relation */
  ByuAPI_FieldDefn *elts;     /* List of field definitions */
  int                length;   /* No. of fields */
} ByuAPI_Signature;

typedef union {    /* Actual Datum fields (of diff sizes) available */
    ByuAPI_Bool        b;
    short   signed int i2;
    short unsigned int c2;
    long    signed int i4;
    long  unsigned int c4;
    float              f4;
    double             f8;
    char               *s;
} ByuAPI_Datum;

typedef struct {		 /* Value of a tuple */
  ByuAPI_Datum    *elts;	 /* Fields in the tuple */
  int length;
  /* The length field is here but it is not being set in different procs. */
} ByuAPI_Tuple;

typedef struct {		 /* Set of tuples */
  ByuAPI_Tuple    *elts;	 /* The actual tuples */
  int             length;        /* No. of tuples */
  int             maximum;       /* Maximum no. of tuples */
} ByuAPI_TupleSet;

typedef struct {
  ByuAPI_Signature sig;	 /* Layout of the relation */
  ByuAPI_TupleSet  tupset;      /* Actual tuples in the relation */
} ByuAPI_Relation;

/********************************/
/*    Opaque data types         */
/********************************/

typedef struct ByuAPI_Session_s *ByuAPI_Session;
typedef struct ByuAPI_SessionState_s *ByuAPI_SessionState;
typedef struct ByuAPI_WriteID_s *ByuAPI_WriteID;

/********************************/
/*      Initialization          */
/********************************/

extern void ByuAPI_Init();

/********************************/
/*      Semantic Guarantees     */
/********************************/

/*
   Guarantees are:
	ARRAY Checktimne of ARRAY ClientSet of ServerSet of Bool
   Checktime = { Read, Write}
   ClientSet = { ReadSetead/read-committed), WroteSet
   ServerSet = { All, Committed}

   The guarantees are represented by 8 bits. You can imagine it as a
   3-dimensional array laid out flat (in row-major order) with the index
   for ServerSet moving the fastest and Checktime moving the slowest.
   (A means compare with server's all, C means committed set)

        Readset        	Writeset        Readset         Writeset
      _____/\_____    _____/\_____    _____/\_____    _____/\______
     /            \  /            \  /            \  /             \

       	A       C       A       C       A       C       A       C
    _________________________________________________________________
    |       |  	    |  	    |  	    |  	    |  	    |  	    |  	    |
    |  MR   |  MR   |  RYW  | RYCW  |  WFR  |  WFR  |  MW   |  MCW  |
    |       |  	    |  	    |  	    |       |  	    |  	    |  	    |
    |       |  	    |  	    |  	    |  WWFR |  	    |  	    |  	    |
    |_______|_______|_______|_______|_______|_______|_______|_______|

     \		                   / \	                           /
      \____________  _____________/   \____________  _____________/
	           \/                              \/
	 Check at Read Time                Check at Write Time
*/

typedef unsigned char ByuAPI_SessionGuarantees;

#define BSG_None   0x00
#define BSG_RRA    1 << 0
#define BSG_RRC    1 << 1
#define BSG_RWA    1 << 2
#define BSG_RWC    1 << 3
#define BSG_WRA    1 << 4
#define BSG_WRC    1 << 5
#define BSG_WWA    1 << 6
#define BSG_WWC    1 << 7
#define BSG_All    0xFF

#define BSG_MCW    BSG_WWC
#define BSG_MW     BSG_WWA
#define BSG_WFCR   BSG_WRC
#define BSG_WWFR   BSG_WRA
#define BSG_WFR    BSG_WWFR | BSG_WFCR
 
#define BSG_RYCW   BSG_RWC
#define BSG_RYW    BSG_RWA
#define BSG_MCR    BSG_RRC
#define BSG_MRmC   BSG_RRA
#define BSG_MR     BSG_MCR | BSG_MRmC


extern ByuAPI_SessionGuarantees ByuAPI_UnionGuarantees(
					ByuAPI_SessionGuarantees g1,
					ByuAPI_SessionGuarantees g2);
/* effects: returns a guarantee which the union of g1 and g2
            i.e, g1 and g2 are both implied by the result.
*/

/********************************/
/*      Session management      */
/********************************/

typedef char * ByuAPI_DatabaseName;

typedef enum {  /* Possible results of a call */
    bbrt_success,      /* Call succeeded */
    bbrt_alloc,        /* Allocation Failure */
    bbrt_invalid      /* Invalid inputs */
} ByuAPI_BasicResultTag;

/* Note that alloc has been added to the API. I think alloc failure
   should be represented in two ways -- return NULL or set this field
   It makes the client code much more neat
*/

typedef struct {
	ByuAPI_BasicResultTag tag;
	union {
	    ByuAPI_Session bbrv_success;   /* Succeeded */
	    /* no more info in invalid cases */
	} u;
} ByuAPI_BeginResult;

extern ByuAPI_BeginResult *
    ByuAPI_BeginSession(ByuAPI_DatabaseName db,
			char *passPhrase,
			ByuAPI_SessionGuarantees sg,
			ByuAPI_Bool committed_only);
/* effects: Starts a session on database "db" with guarantees "sg".
            "passPhrase" should be an ascii phrase that can be used to
	    unlock the current user's public/private key pair.
            If "committed_only" is true, read operations will be performed
	    against the committed database only, else they are to be
	    performed on the complete/tentative database.
            Returns a session handle if it succeeds. Else returns
            appropriate value if the DB is unknown. Returns NULL if there
            is an allocation failure.
*/

typedef struct {
    ByuAPI_BasicResultTag tag;
    union {
	ByuAPI_SessionState bssrv_success;
	/* no more info in invalid case */
    } u;
} ByuAPI_SessionStateResult;
 

extern ByuAPI_SessionStateResult *ByuAPI_GetSessionState(ByuAPI_Session sh);
/* effects: Returns a fresh  copy of the state of session sh. Else returns
            the appropriate value if sh is an invalid handle. Returns NULL
            if there are space allocation problems. Caller is responsible
            for freeing storage using ByuAPI_Free_SessionState.
*/

typedef struct {
    ByuAPI_BasicResultTag tag;
    union {
	char *bsrv_success;
	/* no more info in invalid case */
    } u;
} ByuAPI_StringResult;

extern ByuAPI_StringResult *ByuAPI_UnparseSessionState
						(ByuAPI_SessionState ss);
/* effects: Returns the string encoding of the session state "ss".
            Else returns the appropriate value if ss is an invalid state.
	    Returns NULL if there are space allocation problems. Caller is
	    responsible for freeing storage.
*/

extern ByuAPI_SessionStateResult *ByuAPI_ParseSessionState(char *s);
/* effects: If s encodes a valid session state, returns a a fresh copy of
            the session state. Else returns the appropriate value if s is
	    an invalid encoding. Returns NULL if there are space
	    allocation problems. Caller is responsible for freeing storage
	    using ByuAPI_Free_SessionState. 
*/

extern ByuAPI_SessionStateResult *ByuAPI_SessionStateMax
                       (ByuAPI_SessionState ss1, ByuAPI_SessionState ss2);
/* effects: Allocates fresh storage and returns a session state obtained
	    by merging ss1 and ss2. Else returns the appropriate value if
	    ss1/ss2 are invalid states. Returns NULL if there are space
	    allocation problems. Caller is responsible for freeing
	    storage using ByuAPI_Free_SessionState.
*/

extern ByuAPI_Bool ByuAPI_AddSessionState(ByuAPI_Session sh,
					ByuAPI_SessionState nu);
/* effects: Adds the session state nu to the session sh so that sh is at
            least as consistent as nu. Returns TRUE if it succeeds.
	    Returns FALSE if either sh or nu is invalid.
*/

extern ByuAPI_SessionStateResult *ByuAPI_EndSession(ByuAPI_Session sh);
/* effects: Allocates new storage and returns the state of session sh.
	    Deallocates the storage associated with sh (and sh also). Caller
	    should not use sh for any more calls. Caller is responsible for
	    freeing storage using ByuAPI_Free_SessionState.
	    Else returns the appropriate value if sh is an invalid handle.
	    Returns NULL if there are space allocation problems.
*/

extern int ByuAPI_CommittedIndex(ByuAPI_Signature sig);
extern int ByuAPI_TentativeIndex(ByuAPI_Signature sig);
/* effects: returns the index number of  the committed or tentative field
            in sig. If there is no such field, returns -1.
            These fields use the "b" case of the ByuAPI_Datum union.
*/


/********************************/
/*             Read             */
/********************************/

typedef long int ByuAPI_Seconds;

typedef char *ByuAPI_Query;

typedef enum {        /* Possible results of Read/Write */
    bcrt_success,     /* Operation succeeded */
    bcrt_invalid,     /* Read --> Bad query. */
                      /* Write --> Bad input */
    bcrt_auth,        /* Insufficient authentication */
    bcrt_cant,        /* E.g. Network error or server is not good enough */
    bcrt_alloc        /* Allocation problems. See note at bbrt_alloc */
} ByuAPI_ComplexResultTag;

typedef struct {
  ByuAPI_ComplexResultTag tag;
  union {
    ByuAPI_Relation  brrv_success;	/* Read succeeded */
    char           *brrv_invalid;	/* Bad query      */
    char           *brrv_auth;	        /* Authentication problem */
    char           *brrv_cant;	        /* e.g. network error */
  } u;
} ByuAPI_ReadResult;

extern ByuAPI_ReadResult *ByuAPI_Read(ByuAPI_Session sh,
				   ByuAPI_Query q,
				   ByuAPI_Seconds age,
				   ByuAPI_Seconds maxWait);
/* modifies: sh
   requires: maxWait >=0
   effects: Performs query q in the ByuAPI Query Language according to the
            guarantees being provided in the current session sh. The age
	    parameter specifies that the read is done from a server that
	    contains all writes that were committed more than "age"
	    seconds ago. maxWait is the maximum time that the caller wants to
	    wait for the result.  Returns the a Relation if it succeeds else
	    returns the appropriate error (see ByuAPI_ReadResult above).
	    Returns NULL if there are space allocation problems.
*/

/********************************/
/*             Write            */
/********************************/

typedef struct {		/* Dependency to be check before an
				 * update */
    char *dep_query;	        /* Query to be performed */
    ByuAPI_Relation  dep_expected;	/* Expected result of the query  */
}               ByuAPI_Dependency;

typedef struct {
  ByuAPI_Dependency *elts;	/* List of dependencies */
  int             length;	/* No.  of dependencies */
}               ByuAPI_Dependencies;

typedef enum {
  buo_insert, buo_delete, buo_modify
}               ByuAPI_UpdateOp;

typedef struct {
  ByuAPI_RelationName bu_reln;	/* Relation to perform data on */
  ByuAPI_UpdateOp  bu_op;	/* Type of operation           */
  ByuAPI_Relation  bu_tuples;	/* Perform op on these tuples  */
}               ByuAPI_Update;

typedef struct {
  ByuAPI_Update   *elts;		/* List of updates */
  int             length;	/* No.  of updates */
}               ByuAPI_Updates;

typedef struct {
  ByuAPI_ComplexResultTag tag;
  union {
    ByuAPI_WriteID   bwrv_success;
    char           *bwrv_invalid;
    char           *bwrv_auth;	/* Authentication problem */
    char           *bwrv_cant;	/* e.g. connectivity too poor */
  }               u;
}               ByuAPI_WriteResult;

extern ByuAPI_WriteResult *ByuAPI_Write(ByuAPI_Session sh,
				      ByuAPI_Dependencies deps,
				      ByuAPI_Updates updates,
				      char *mergeproc,
				      ByuAPI_Seconds maxWait);
/* modifies: sh
   effects: Checks if the predicate specified by "deps" is true or not
	    (i.e. runs the query and checks the expected results).  If the
	    check succeeds, the modifications given by "updates" are
	    applied to the tentative database. If the check fails the
	    "mergeproc", written in the MergeProc language, is executed.
	    maxWait is the maximum time that the caller wants to wait for
	    the result.  If the write succeeds, returns a writeid for this
	    write. Else returns the appropriate error code (see
	    ByuAPI_WriteResult above) Returns NULL if there are space
	    allocation problems. Caller is responsible for freeing storage using
	    ByuAPI_Free_WriteID.
*/


extern ByuAPI_Bool ByuAPI_IsWriteCommitted(ByuAPI_Session sh, 
					   ByuAPI_WriteID wid,  
					   ByuAPI_Seconds maxWait);
/* requires: WriteID to be a valid write ID (i.e. has to be either
             tentative or committed). sh is a valid session
   effects:  Returns true iff the possibility that wid is tentative
             can be ruled out in the given amount of time for the session sh.
*/


/* Procedures for freeing various data structures */


extern ByuAPI_Bool ByuAPI_Free_SessionState(ByuAPI_SessionState st);
/* effects: for a valid st, frees the storage associated with st and
            returns TRUE; otherwise, returns FALSE. Frees st also.
*/

extern ByuAPI_Bool ByuAPI_Free_WriteID(ByuAPI_WriteID wid);
/* effects: Given a valid id, wid, frees the storage associated with it
            and returns TRUE. Else returns FALSE for an invalid id.
	    Frees wid also.
*/

extern ByuAPI_Bool ByuAPI_Free_BeginResult (ByuAPI_BeginResult *bbr);
/* effects: Given a valid pointer to a beginresult, frees the storage
            associated with it and returns TRUE. Else returns FALSE.
	    Frees bbr also. Does not operate on the session
*/

extern ByuAPI_Bool ByuAPI_Free_SessionStateResult 
                               (ByuAPI_SessionStateResult *bssr);
/* effects: Given a valid pointer to a sessionstateresult, frees the storage
            associated with it and returns TRUE. Else returns FALSE.
	    Frees bssr also.
*/

extern ByuAPI_Bool ByuAPI_Free_StringResult (ByuAPI_StringResult *bsr);
/* effects: Given a valid pointer to a stringresult, frees the storage
            associated with it and returns TRUE. Else returns FALSE.
	    Frees bsr also.
*/

extern ByuAPI_Bool ByuAPI_Free_ReadResult (ByuAPI_ReadResult *brr);
/* effects: Given a valid pointer to a readresult, frees the storage
            associated with it and returns TRUE. Else returns FALSE.
	    Frees brr also.
*/

extern ByuAPI_Bool ByuAPI_Free_WriteResult (ByuAPI_WriteResult *bwr);
/* effects: Given a valid pointer to a writeresult, frees the storage
            associated with it and returns TRUE. Else returns FALSE.
	    Frees bwr also.
*/


#endif /* def(_BYUAPI_H) */
