#ifndef _BLURB_
#define _BLURB_
/*

            Coda: an Experimental Distributed File System
                             Release 3.1

          Copyright (c) 1987-1995 Carnegie Mellon University
                         All Rights Reserved

Permission  to  use, copy, modify and distribute this software and its
documentation is hereby granted,  provided  that  both  the  copyright
notice  and  this  permission  notice  appear  in  all  copies  of the
software, derivative works or  modified  versions,  and  any  portions
thereof, and that both notices appear in supporting documentation, and
that credit is given to Carnegie Mellon University  in  all  documents
and publicity pertaining to direct or indirect use of this code or its
derivatives.

CODA IS AN EXPERIMENTAL SOFTWARE SYSTEM AND IS  KNOWN  TO  HAVE  BUGS,
SOME  OF  WHICH MAY HAVE SERIOUS CONSEQUENCES.  CARNEGIE MELLON ALLOWS
FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.   CARNEGIE  MELLON
DISCLAIMS  ANY  LIABILITY  OF  ANY  KIND  FOR  ANY  DAMAGES WHATSOEVER
RESULTING DIRECTLY OR INDIRECTLY FROM THE USE OF THIS SOFTWARE  OR  OF
ANY DERIVATIVE WORK.

Carnegie  Mellon  encourages  users  of  this  software  to return any
improvements or extensions that  they  make,  and  to  grant  Carnegie
Mellon the rights to redistribute these changes without encumbrance.
*/

static char *rcsid = "$Header: fso_cachefile.c,v 1.3.1.1 95/10/11 10:25:07 raiff Exp $";
#endif /*_BLURB_*/




/*
 *    Cache file management
 */

#ifdef __cplusplus
extern "C" {
#endif __cplusplus

#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <errno.h>
#include <stdio.h>
#include <setjmp.h>
#include <sysent.h>

extern void compress();
extern void decompress();

#ifdef __cplusplus
}
#endif __cplusplus

/* interfaces */
#include <vice.h>

/* from venus */
#include "fso.h"
#include "venus.private.h"


/*  *****  CacheFile Members  *****  */

/* Pre-allocation routine. */
/* MUST be called from within transaction! */
CacheFile::CacheFile(int i) {
    ASSERT(this	!= 0);

    /* Assume caller has done RVMLIB_REC_OBJECT! */
/*    RVMLIB_REC_OBJECT(*this);*/
    sprintf(name, "V%d", i);
    inode = (ino_t)-1;
    compressed = 0;
    length = 0;

    /* Container reset will be done by eventually by FSOInit()! */
}


CacheFile::CacheFile() {
    if (Simulating) return;

    ASSERT(inode != (ino_t)-1 && !compressed && length == 0);
}


CacheFile::~CacheFile() {
    if (Simulating) return;

    ASSERT(inode != (ino_t)-1 && !compressed && length == 0);
}


/* MUST NOT be called from within transaction! */
void CacheFile::Validate() {
    if (!ValidContainer())
	ResetContainer();
}


/* MUST NOT be called from within transaction! */
void CacheFile::Reset() {
    if (inode == (ino_t)-1 || compressed || length != 0 || !ValidContainer())
	ResetContainer();
}


int CacheFile::ValidContainer() {
    if (Simulating) return(1);

    int code = 0;
    struct stat tstat;
    int valid = (code = ::stat(name, &tstat)) == 0 &&
      tstat.st_uid == (uid_t)V_UID &&
      tstat.st_gid == (gid_t)V_GID &&
      (tstat.st_mode & ~S_IFMT) == V_MODE &&
      tstat.st_ino == inode &&
      tstat.st_size == length;

    if (!valid && LogLevel >= 10) {
	dprint("CacheFile::ValidContainer: %s invalid\n", name);
	if (code == 0)
	    dprint("\t(%u, %u), (%u, %u), (%o, %o), (%d, %d), (%d, %d)\n",
		   tstat.st_uid, (uid_t)V_UID, tstat.st_gid, (gid_t)V_GID,
		   (tstat.st_mode & ~S_IFMT), V_MODE,
		   tstat.st_ino, inode, tstat.st_size, length);
	else
	    dprint("\tstat failed (%d)\n", errno);
    }

    return(valid);
}


/* MUST NOT be called from within transaction! */
void CacheFile::ResetContainer() {
    if (Simulating) return;

    LOG(10, ("CacheFile::ResetContainer: %s, %d, %d, %d\n",
	      name, inode, compressed, length));

    int tfd;
    struct stat tstat;
    if ((tfd = ::open(name, O_RDWR | O_CREAT | O_TRUNC, V_MODE)) < 0)
	Choke("CacheFile::ResetContainer: open failed (%d)", errno);
    if (::fchmod(tfd, V_MODE) < 0)
	Choke("CacheFile::ResetContainer: fchmod failed (%d)", errno);
    if (::fchown(tfd, (uid_t)V_UID, (gid_t)V_GID) < 0)
	Choke("CacheFile::ResetContainer: fchown failed (%d)", errno);
    if (::fstat(tfd, &tstat) < 0)
	Choke("CacheFile::ResetContainer: fstat failed (%d)", errno);
    if (::close(tfd) < 0)
	Choke("CacheFile::ResetContainer: close failed (%d)", errno);

    ATOMIC(
	RVMLIB_REC_OBJECT(*this);
	inode = tstat.st_ino;
	compressed = 0;
	length = 0;
    , MAXFP)
}


/*
 * moves a cache file to a new one.  destination's name already
 * there, created by cache file constructor.
 */  
void CacheFile::Move(CacheFile *destination) {
    if (Simulating) return;

    LOG(10, ("CacheFile::Move: source: %s, %d %d, %d, dest: %s\n",
	      name, inode, compressed, length, destination->name));

    destination->inode = inode;
    destination->compressed = compressed;
    destination->length = length;
    if (::rename(name, destination->name) != 0)
        Choke("CacheFile::RenameContainer: rename failed (%d)", errno);
}


/* 
 * copies a cache file, data and attributes, to a new one.  
 */
void CacheFile::Copy(CacheFile *source) {
    if (Simulating) return;

    LOG(10, ("CacheFile::Copy: %s, %d, %d, %d\n",
	      name, inode, compressed, length));

    int tfd, ffd, n;
    struct stat tstat;
    char buf[MAXBSIZE];

    if ((tfd = ::open(name, O_RDWR | O_CREAT | O_TRUNC, V_MODE)) < 0)
	Choke("CacheFile::Copy: open failed (%d)", errno);
    if (::fchmod(tfd, V_MODE) < 0)
	Choke("CacheFile::Copy: fchmod failed (%d)", errno);
    if (::fchown(tfd, (uid_t)V_UID, (gid_t)V_GID) < 0)
	Choke("CacheFile::Copy: fchown failed (%d)", errno);

    if ((ffd = ::open(source->name, O_RDONLY, V_MODE)) < 0)
	Choke("CacheFile::Copy: source open failed (%d)", errno);

    for (;;) {
        n = ::read(ffd, buf, (int) sizeof(buf));
        if (n == 0)
	    break;
        if (n < 0)
	    Choke("CacheFile::Copy: read failed! (%d)", errno);
	if (::write(tfd, buf, n) != n)
	    Choke("CacheFile::Copy: write failed! (%d)", errno);
    }
    if (::fstat(tfd, &tstat) < 0)
	Choke("CacheFile::Copy: fstat failed (%d)", errno);
    if (::close(tfd) < 0)
	Choke("CacheFile::Copy: close failed (%d)", errno);
    if (::close(ffd) < 0)
	Choke("CacheFile::Copy: source close failed (%d)", errno);
    
    ASSERT(source->length == tstat.st_size);

    inode = tstat.st_ino;
    compressed = source->compressed;
    length = source->length;
}


void CacheFile::Remove() {
    compressed = 0;
    length = 0;
    if (::unlink(name) < 0)
        Choke("CacheFile::Remove: unlink failed (%d)", errno);
}


/* N.B. length member is NOT updated as side-effect! */
void CacheFile::Stat(struct stat *tstat) {
    if (Simulating) return;

    ASSERT(inode != (ino_t)-1 && !compressed);

    ASSERT(::stat(name, tstat) == 0);
    ASSERT(tstat->st_ino == inode);
}


/* MUST be called from within transaction! */
void CacheFile::Truncate(unsigned newlen) {
    if (Simulating) return;

    ASSERT(inode != (ino_t)-1 && (!compressed || newlen == 0));

    if (compressed || length != newlen) {
	RVMLIB_REC_OBJECT(*this);
	compressed = 0;
	length = newlen;
    }

    ASSERT(::truncate(name, length) == 0);
}


/* MUST be called from within transaction! */
void CacheFile::SetLength(unsigned newlen) {
    ASSERT(inode != (ino_t)-1 && !compressed);

    if (length != newlen) {
	 RVMLIB_REC_OBJECT(*this);
	length = newlen;
    }
}


/* MUST NOT be called from within transaction! */
int CacheFile::Compress() {
    ASSERT(inode != (ino_t)-1 && !compressed);
    return(CompressDecompressCommon(1));
}


/* MUST NOT be called from within transaction! */
int CacheFile::Uncompress() {
    ASSERT(inode != (ino_t)-1 && compressed);
    return(CompressDecompressCommon(0));
}


jmp_buf VenusCompressEnv;
PRIVATE	char compress_sync = 0;	    /* only one compress/uncompress allowed at a time! */
PRIVATE int compress_waiters = 0;

/* MUST NOT be called from within transaction! */
int CacheFile::CompressDecompressCommon(int docompress) {
    ASSERT(!Simulating);

    /* Grab semaphore. */
    while (compress_sync != 0) {
	compress_waiters++;
	VprocWait(&compress_sync);
	compress_waiters--;
    }
    compress_sync = 1;

    /* Set up stdin and stdout. */
    ASSERT(freopen(name, "r", stdin) != NULL);
    char tmpfile[MAXNAMLEN];
    sprintf(tmpfile, "%s.tmp", name);
    ASSERT(freopen(tmpfile, "w+", stdout) != NULL);
    ASSERT(fchown(fileno(stdout), (uid_t)V_UID, (gid_t)V_GID) == 0);
    ASSERT(fchmod(fileno(stdout), V_MODE) == 0);

    int code = 0;
    if (setjmp(VenusCompressEnv) == 0) {
	/* Attempt the compress/decompress. */
	if (docompress)
	    compress();
	else
	    decompress();
	ASSERT(!ferror(stdin));
	(void)fclose(stdin);
	ASSERT(!ferror(stdout));
	(void)fclose(stdout);

	/* Rename the tmp file and record the fact that this cache file is compressed/decompressed. */
	ASSERT(rename(tmpfile, name) == 0);
	struct stat tstat;
	ASSERT(::stat(name, &tstat) == 0);
	ATOMIC(
	    RVMLIB_REC_OBJECT(*this);
	    inode = tstat.st_ino;
	    compressed = docompress;
	    length = tstat.st_size;
	, MAXFP)
    }
    else {
	/* Failure.  Discard the tmp file. */
	(void)fclose(stdin);
	(void)fclose(stdout);
	unlink(tmpfile);

	code = ENOSPC;		    /* could have been EIO instead! */
    }

    /* Release semaphore. */
    compress_sync = 0;
    if (compress_waiters > 0)
	VprocSignal(&compress_sync);

    return(code);
}


void CacheFile::print(int fdes) {
    fdprint(fdes, "[ %s, %d, %d, %d ]\n",
	     name, inode, compressed, length);
}

