#ifndef _BLURB_
#define _BLURB_
/*

            Coda: an Experimental Distributed File System
                             Release 3.1

          Copyright (c) 1987-1995 Carnegie Mellon University
                         All Rights Reserved

Permission  to  use, copy, modify and distribute this software and its
documentation is hereby granted,  provided  that  both  the  copyright
notice  and  this  permission  notice  appear  in  all  copies  of the
software, derivative works or  modified  versions,  and  any  portions
thereof, and that both notices appear in supporting documentation, and
that credit is given to Carnegie Mellon University  in  all  documents
and publicity pertaining to direct or indirect use of this code or its
derivatives.

CODA IS AN EXPERIMENTAL SOFTWARE SYSTEM AND IS  KNOWN  TO  HAVE  BUGS,
SOME  OF  WHICH MAY HAVE SERIOUS CONSEQUENCES.  CARNEGIE MELLON ALLOWS
FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.   CARNEGIE  MELLON
DISCLAIMS  ANY  LIABILITY  OF  ANY  KIND  FOR  ANY  DAMAGES WHATSOEVER
RESULTING DIRECTLY OR INDIRECTLY FROM THE USE OF THIS SOFTWARE  OR  OF
ANY DERIVATIVE WORK.

Carnegie  Mellon  encourages  users  of  this  software  to return any
improvements or extensions that  they  make,  and  to  grant  Carnegie
Mellon the rights to redistribute these changes without encumbrance.
*/

static char *rcsid = "$Header: cfs_mach.c,v 3.3.2.1 95/10/11 10:03:25 raiff Exp $";
#endif /*_BLURB_*/

/* ************************************************** */
/* These routines define the psuedo device for communication between
 * Coda's Venus and Minicache in Mach 2.6. They used to be in cfs_subr.c, 
 * but I moved them to make it easier to port the Minicache without 
 * porting coda. -- DCS 10/12/94
 */

/* These routines are the device entry points for Venus. */

extern int cfsnc_initialized;    /* Set if cache has been initialized */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <cfs/cfs.h>
#include <cfs/cfsio.h>

struct vnode *cfs_ctlvp = 0;

int
vcopen(dev)
        dev_t dev;
{
	register struct vcomm *vcp;
	struct ody_mntinfo *op;

	CFSDEBUG(CFS_ROOT, printf("vcopen(%d)\n", minor(dev)); );
	
	if (minor(dev) >= NMACH_CFS || minor(dev) < 0)
	    return(ENXIO);

	if (!cfsnc_initialized)
	  cfsnc_init();

	if (cfs_ctlvp == 0) {
	    ViceFid ctlfid;
	    
	    ctlfid.Volume = CTL_VOL;
	    ctlfid.Vnode = CTL_VNO;
	    ctlfid.Unique = CTL_UNI;
	    cfs_ctlvp = CTOV(makecfsnode(&ctlfid, 0, VCHR));
	}
	
	vcp = &cfs_mnttbl[minor(dev)].mi_vcomm;
	if (VC_OPEN(vcp))
	    return(EBUSY);

	/* Make first 4 bytes be zero */
	cfs_mnttbl[minor(dev)].mi_name = (char *)0;
	vcp->vc_selproc = 0;
	INIT_QUEUE(vcp->vc_requests);
	INIT_QUEUE(vcp->vc_replys);
	MARK_VC_OPEN(vcp);

	cfs_mnttbl[minor(dev)].mi_vfschain.vfsp = NULL;
	cfs_mnttbl[minor(dev)].mi_vfschain.rootvp = NULL;
	cfs_mnttbl[minor(dev)].mi_vfschain.next = NULL;

	return(0);
}


int
vcclose(dev)
	dev_t dev;
{
	register struct vcomm *	vcp;
	register struct vmsg *vmp;
	struct queue save;	
	struct ody_mntinfo *op;
	
	CFSDEBUG(CFS_ROOT, printf("vcclose(%d)\n", minor(dev)); );

	if (minor(dev) >= NMACH_CFS || minor(dev) < 0)
	    return(ENXIO);
	
	vcp = &cfs_mnttbl[minor(dev)].mi_vcomm;

	if (!VC_OPEN(vcp))
	    panic("vcclose: not open");

	if (cfs_mnttbl[minor(dev)].mi_name) {
	    kfree(cfs_mnttbl[minor(dev)].mi_name,
		  strlen(cfs_mnttbl[minor(dev)].mi_name));
	    cfs_mnttbl[minor(dev)].mi_name = 0;
	}
	
	/* prevent future operations on this vfs from succeeding by auto-
	 * unmounting any vfs mounted via this device. This frees user or
	 * sysadm from having to remember where all mount points are located.
	 * Put this before WAKEUPs to avoid queuing new messages between
	 * the WAKEUP and the unmount (which can happen if we're unlucky)
	 */
	for (op = &cfs_mnttbl[minor(dev)].mi_vfschain; op ; op = op->next) {
	    if (op->rootvp) {
		VTOC(op->rootvp)->c_flags |= C_DYING;	/* Let unmount know this is for real */
		u.u_error = 0;	/* dounmount uses u_error to relay return codes */

		dounmount(op->vfsp);	/* defined in vfs/vfs.c */
		if (u.u_error)
		    printf("Error %d unmounting vfs in vcclose(%d)\n", u.u_error, minor(dev));
	    } else {
		/* Should only be null if no mount has happened. */
		if (op != &cfs_mnttbl[minor(dev)].mi_vfschain) 
		    printf("Help! assertion failed in vcwrite\n");

	    }
	}

	/* Wakeup clients so they can return. Note: This trashes the queue. */
	for (vmp = (struct vmsg *)GETNEXT(vcp->vc_requests);
	     !EOQ(vmp, vcp->vc_requests);
	     vmp = (struct vmsg *)GETNEXT(save))
	    {
		save = vmp->vm_chain;	/* the free can trash vmp, save ptr to next req */
		
		/* Free signal request messages and don't wakeup cause no one is waiting. */
		if (vmp->vm_opcode == CFS_SIGNAL) {
		    kfree((caddr_t)vmp->vm_data, (u_int)VC_IN_NO_DATA);
		    kfree((caddr_t)vmp, (u_int)sizeof(struct vmsg));
		    continue;
		}

		WAKEUP(&vmp->vm_sleep);
	    }

	for (vmp = (struct vmsg *)GETNEXT(vcp->vc_replys);
	     !EOQ(vmp, vcp->vc_replys);
	     vmp = (struct vmsg *)GETNEXT(vmp->vm_chain))
	    {
		WAKEUP(&vmp->vm_sleep);
	    }
	    
	MARK_VC_CLOSED(vcp);
}


int
vcread(dev, uiop)
	dev_t dev;
	struct uio *uiop;
{
	register struct vcomm *	vcp;
	register struct vmsg *vmp;
	int error = 0;

	if (minor(dev) >= NMACH_CFS || minor(dev) < 0)
	    return(ENXIO);
	
	vcp = &cfs_mnttbl[minor(dev)].mi_vcomm;
	/* Get message at head of request queue. */
	if (EMPTY(vcp->vc_requests))
	    return(0);	/* Nothing to read */
	
	vmp = (struct vmsg *)GETNEXT(vcp->vc_requests);

	/* Move the input args into userspace */
	error = uiomove(vmp->vm_data, vmp->vm_inSize, UIO_READ, uiop);
	if (error) {
	    printf("vcread: error (%d) on uiomove\n", error);
	    error = EINVAL;
	}

	REMQUE(vmp->vm_chain);

	/* If request was a signal, free up the message and don't enqueue it in the reply queue. */
	if (vmp->vm_opcode == CFS_SIGNAL) {
	    if (cfsdebug)
		printf("vcread: signal msg (%d, %d)\n", vmp->vm_opcode, vmp->vm_unique);

	    kfree((caddr_t)vmp->vm_data, (u_int)VC_IN_NO_DATA);
	    kfree((caddr_t)vmp, (u_int)sizeof(struct vmsg));
	    return(error);
	}

	vmp->vm_flags |= VM_READ;
	INSQUE(vmp->vm_chain, vcp->vc_replys);

	return(error);
}
#ifdef SUN4C
#else
extern int handleDownCall(int opcode, struct outputArgs *out);
#endif SUN4C

int
vcwrite(dev, uiop)
	dev_t dev;
	struct uio *uiop;
{
	register struct vcomm *	vcp;
	register struct vmsg *vmp;
	struct outputArgs *out;
	u_long seq;
	u_long opcode;
	int buf[2];
	int error = 0;

	if (minor(dev) >= NMACH_CFS || minor(dev) < 0)
	    return(ENXIO);
	
	vcp = &cfs_mnttbl[minor(dev)].mi_vcomm;

	/* Peek at the opcode, unique without transfering the data. */
	error = uiomove(buf, sizeof(int) * 2, UIO_WRITE, uiop);
	if (error) {
	    printf("vcwrite: error (%d) on uiomove\n", error);
	    return(EINVAL);
	}

	opcode = buf[0];
	seq = buf[1];
	
	if (cfsdebug)
	    printf("vcwrite got a call for %d.%d\n", opcode, seq);

	if (DOWNCALL(opcode)) {
	    struct outputArgs pbuf;

	    /* get the rest of the data. */
	    error = uiomove(&pbuf.result, sizeof(pbuf) - (sizeof(int)*2), UIO_WRITE, uiop);
	    if (error) {
		printf("vcwrite: error (%d) on uiomove (Op %d seq %d)\n", error, opcode, seq);
		return(EINVAL);
	    }
	    
	    return handleDownCall(opcode, &pbuf);
	}

	/* Look for the message on the (waiting for) reply queue. */
	for (vmp = (struct vmsg *)GETNEXT(vcp->vc_replys);
	     !EOQ(vmp, vcp->vc_replys);
	     vmp = (struct vmsg *)GETNEXT(vmp->vm_chain))
	    {
		if (vmp->vm_unique == seq) break;
	    }
	
	if (EOQ(vmp, vcp->vc_replys)) {
	    if (cfsdebug)
		printf("vcwrite: msg (%d, %d) not found\n", opcode, seq);

	    return(ESRCH);
	}

	/* Remove the message from the reply queue */
	REMQUE(vmp->vm_chain);

	/* move data into response buffer. */
	out = (struct outputArgs *)vmp->vm_data;
	/* Don't need to copy opcode and uniquifier. */
	
	/* get the rest of the data. */
	if (vmp->vm_outSize < uiop->uio_resid) {
	    printf("vcwrite: more data than asked for (%d < %d)\n",
		   vmp->vm_outSize, uiop->uio_resid);
	    WAKEUP(&vmp->vm_sleep); 	/* Notify caller of the error. */
	    return(EINVAL);
	} 

	buf[0] = uiop->uio_resid; 	/* Save this value. */
	error = uiomove(&out->result, vmp->vm_outSize - (sizeof(int) * 2),
			UIO_WRITE, uiop);
	if (error) {
	    printf("vcwrite: error (%d) on uiomove (op %d seq %d)\n", error, opcode, seq);
	    return(EINVAL);
	}

	out->opcode = opcode;		/* I don't think these are used, but just in case. */
	out->unique = seq;
	vmp->vm_outSize	= buf[0];	/* Amount of data transferred? */
	vmp->vm_flags |= VM_WRITE;
	WAKEUP(&vmp->vm_sleep);

	return(0);
}

extern int cfsnc_use;

int
vcioctl(dev,cmd,addr,flag)
     dev_t dev;
     caddr_t addr;
{
  switch(cmd) {
    case CFSRESIZE : {
      struct cfs_resize *data = (struct cfs_resize *)addr;
      return(cfsnc_resize(data->hashsize, data->heapsize));
    }

    case CFSSTATS :
      if (cfsnc_use) {
        cfsnc_gather_stats();
        return(0);
      } else {
	return(ENODEV);
      }

    case CFSPRINT :
      if (cfsnc_use) {
	      print_cfsnc();
	      return(0);
      } else {
	return(ENODEV);
      }

    case ODYBIND:
      /* Bind a name to our device. Used to allow more than one kind of FS */
      {
	  if (cfs_mnttbl[minor(dev)].mi_name) {
	      if (cfsdebug)
		  printf("ODYBIND: dev %d already has name %s\n", minor(dev),
			 cfs_mnttbl[minor(dev)].mi_name);
	      return(EBUSY);	/* Some name already used. */
	  } else {
	      struct ody_bind *data = (struct ody_bind *)addr;
	      char *name = (char *)kalloc(data->size);
	      
	      copyin(data->name, name, data->size);
	      
	      if (cfsdebug)
		  printf("ODYBIND: binding %s to dev %d\n", name, minor(dev));

	      cfs_mnttbl[minor(dev)].mi_name = name;
	      return(0);
	  }
      }

    case ODYPROD:	/* do a wakeup on the specified address. */
      {
	  printf("addr %x\n", *(int *)addr);
	  wakeup(*(int *)addr);
	  return(0);
      }
      
    case ODYREQ:	/* print out all requests pending to a warden. */
      {
	  register struct vcomm *vcp;
	  register struct vmsg *vmp;
	  int m = *(int *)addr;
	  printf("m %d\n", m);

	  if (m >= NMACH_CFS || m < 0)
	      return(ENXIO);

	  vcp = &cfs_mnttbl[m].mi_vcomm;
	  for (vmp = (struct vmsg *)GETNEXT(vcp->vc_replys);
	       !EOQ(vmp, vcp->vc_replys);
	       vmp = (struct vmsg *)GETNEXT(vmp->vm_chain))
	      {
		  printf("CFS%d: op (%d.%d) flags %x insize %d outsize %d\n",
			 m, vmp->vm_opcode, vmp->vm_unique, vmp->vm_flags,
			 vmp->vm_inSize, vmp->vm_outSize);
	      }
	  return(0);
      }

    default :
      return(EINVAL);
    }
}

int
vcselect(dev, flag)
	dev_t dev;
	int flag;
{
	register struct vcomm *	vcp;

	if (minor(dev) >= NMACH_CFS || minor(dev) < 0)
	    return(ENXIO);
	
	vcp = &cfs_mnttbl[minor(dev)].mi_vcomm;

	if (flag != FREAD)
	    return(0);

	if (!EMPTY(vcp->vc_requests))
	    return(1);

	vcp->vc_selproc = (SELPROC)current_thread();
	return(0);
}

