#ifndef _BLURB_
#define _BLURB_
/*

            Coda: an Experimental Distributed File System
                             Release 3.1

          Copyright (c) 1987-1995 Carnegie Mellon University
                         All Rights Reserved

Permission  to  use, copy, modify and distribute this software and its
documentation is hereby granted,  provided  that  both  the  copyright
notice  and  this  permission  notice  appear  in  all  copies  of the
software, derivative works or  modified  versions,  and  any  portions
thereof, and that both notices appear in supporting documentation, and
that credit is given to Carnegie Mellon University  in  all  documents
and publicity pertaining to direct or indirect use of this code or its
derivatives.

CODA IS AN EXPERIMENTAL SOFTWARE SYSTEM AND IS  KNOWN  TO  HAVE  BUGS,
SOME  OF  WHICH MAY HAVE SERIOUS CONSEQUENCES.  CARNEGIE MELLON ALLOWS
FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.   CARNEGIE  MELLON
DISCLAIMS  ANY  LIABILITY  OF  ANY  KIND  FOR  ANY  DAMAGES WHATSOEVER
RESULTING DIRECTLY OR INDIRECTLY FROM THE USE OF THIS SOFTWARE  OR  OF
ANY DERIVATIVE WORK.

Carnegie  Mellon  encourages  users  of  this  software  to return any
improvements or extensions that  they  make,  and  to  grant  Carnegie
Mellon the rights to redistribute these changes without encumbrance.
*/

static char *rcsid = "$Header: blurb.c,v 3.2.2.1 95/10/11 10:03:02 raiff Exp $";
#endif /*_BLURB_*/









#ifdef __cplusplus
extern "C" {
#endif __cplusplus

#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <libc.h>
#include <errno.h>
#include <strings.h>

#ifdef __cplusplus
}
#endif __cplusplus

#include <util.h>

#define FIRSTLINE "#ifndef _BLURB_\n"
#define FIRSTLINE_RPC "%{#ifndef _BLURB_\n"
#define FIRSTLINE_LEXYACC "%{\n"
#define SECONDLINE "#define _BLURB_\n"
#define SECONDLINE_LEXYACC "#ifndef _BLURB_\n#define _BLURB_\n"
#define LASTLINE "#endif /*_BLURB_*/\n"
#define LASTLINE_RPC "#endif /*_BLURB_*/%}\n"

#define MAXLINELEN 1024  /* max len of line in input file */

int NoRenameFlag;  /* if set to 1, original file is not removed */
int VerboseFlag;    /* Set to one to hear ball by ball description */


char *BlurbFile = "/usr/coda/include/blurb.doc";;  


/* set based on file name */
enum {CFILE, MAKEFILE, ASMFILE, RPCFILE, LEXFILE, YACCFILE, UNKNOWN} FileType;  


/* Function decls */
void ProcessOneFile(char *name);
int FixFile(FILE *fin, FILE *fout, FILE *fblurb);
int EatArgs(int argc, char *argv[]);
void DetermineFileType(char *name);


main(int argc, char *argv[])
    {
    register int i, file1;

    file1 = EatArgs(argc, argv);
    
    for (i=file1; i < argc; i++)
	{
	ProcessOneFile(argv[i]);
	}
    }


void ProcessOneFile(char *name)
    {
    FILE *fin, *fout, *fblurb;
    char tmpfile[MAXPATHLEN];

    fin = fout= fblurb = 0;

    DetermineFileType(name);
    if (FileType == UNKNOWN)
	{
	printf("%s: unknown file type ... ignoring\n", name);
	goto QUIT;
	}

    /* Play safe initially */
    fblurb = fopen (BlurbFile, "r");
    if (!fblurb) {perror(BlurbFile); goto QUIT;}

    fin = fopen (name, "r");
    if (!fin) {perror(name); goto QUIT;}
    
    if (SafeStrCpy(tmpfile, name, sizeof(tmpfile))) goto QUIT;
    if (SafeStrCat(tmpfile, ".b", sizeof(tmpfile))) goto QUIT;
    
    fout = fopen (tmpfile, "w");
    if (!fout) {perror(tmpfile); goto QUIT;}

    if (VerboseFlag) printf("%s ...", name);
    if (FixFile(fin, fout, fblurb)) goto QUIT;
    
    if (NoRenameFlag) goto QUIT;

    /* Now do risky part -- here's where transactions would help! */
    if (unlink(name)) {perror(name); goto QUIT;}
    if (rename(tmpfile, name)) {perror(tmpfile); goto QUIT;}

QUIT:
    if (fin) fclose(fin);
    if (fout) fclose(fout);
    if (fblurb) fclose(fblurb);
    }

int FixFile(FILE *fin, FILE *fout, FILE *fblurb)
    {
    char nextline[MAXLINELEN], blurbline[MAXLINELEN];
    char *s, *firstline, *secondline, *lastline;

    /* Strip off existing blurb, if any */
    s = fgets(nextline, MAXLINELEN, fin);
    if (!s)
        {
	printf("Premature end of file\n");
	printf("Check for '#endif /*_BLURB_*/', should be '#endif /*_BLURB_*/'\n");
	printf("Run blurb-ansify to fix this\n");
	return(-1);
        }

    switch (FileType)
	{
	case RPCFILE:
	                firstline = FIRSTLINE_RPC;
			secondline = SECONDLINE;
			lastline = LASTLINE_RPC;
			break;
			
	case LEXFILE:
	case YACCFILE:
	                firstline = FIRSTLINE_LEXYACC;
			secondline = SECONDLINE_LEXYACC;
			lastline = LASTLINE;
			break;
			
	default:	firstline = FIRSTLINE;
			secondline = SECONDLINE;
			lastline = LASTLINE;
			break;
		
	}
    
    if (strcmp(nextline, firstline) == 0) 
	{
	do 
	    {
	    s = fgets(nextline, MAXLINELEN, fin);
	    if (!s) {printf("Premature end of file\n"); return(-1);}
	    }
	while (strcmp(nextline, lastline) != 0);
	s = fgets(nextline, MAXLINELEN, fin); 
	if (!s) nextline[0] = 0;  /* so writing it out will be ok */
	}
    
    /* At this point nextline contains the first non-blurb line of the input */

    /* Write preamble */
    fputs(firstline, fout);
    fputs(secondline, fout);
    if (FileType == ASMFILE) fputs("#ifdef undef\n", fout);

    /* Copy blurb */
    for(s = fgets(blurbline, MAXLINELEN, fblurb); s != NULL;
		s = fgets(blurbline, MAXLINELEN, fblurb))
	{
	if (FileType == MAKEFILE) fputc('#', fout);
	fputs(blurbline, fout);
	}
    
    
    /* Write epilogue */
    if (FileType == ASMFILE) fputs("#endif undef\n", fout);
    fputs(lastline, fout);
    fputc('\n', fout);

    /* Copy rest of input file to output */
    do
	{
	fputs(nextline, fout);
	s = fgets(nextline, MAXLINELEN, fin);
	}
    while (s);
    
    if (VerboseFlag) printf("OK\n");
    return(0);
    }


int EatArgs(int argc, char *argv[])
    /* Returns index of first uneaten arg in argv[] */
    {
    register int i;

    if (argc == 1) goto BadArgs;

    for (i = 1; i < argc; i++)
	{
	if (strcmp(argv[i], "-n") == 0)
	    {
	    NoRenameFlag = 1;
	    continue;
	    }

	if (strcmp(argv[i], "-v") == 0)
	    {
	    VerboseFlag = 1;
	    continue;
	    }

	if (strcmp(argv[i], "-f") == 0 && i < argc)
	    {
	    BlurbFile = argv[++i];
	    continue;
	    }

	if (*argv[i] == '-') goto BadArgs;
	return(i);
	}
    
    BadArgs:
    fprintf(stderr, "Usage: blurb [-v] [-n] [-f blurbfile] file1 file2 ....\n");
    exit(-1);
    }




void DetermineFileType(char *name)
    {
    int ll;

    ll = strlen(name);


    /* Have to check for .lex and .yacc endings before makefile for SynRGen sources */
    if (ll > 3 && strcmp(&name[ll-4], ".lex") == 0)
	{
	FileType = LEXFILE;
	return;
        }

    if (ll > 4 && strcmp(&name[ll-5], ".yacc") == 0)
	{
	FileType = YACCFILE;
	return;
        }

    if (strncmp(name, "Makefile", strlen("Makefile")) == 0  
	    || strncmp(name, "makefile", strlen("makefile")) == 0
	    || strncmp(name, "Makeconf", strlen("Makeconf")) == 0)
	{
	FileType = MAKEFILE;
	return;
	}

    if (ll > 1 && name[ll-2] == '.' && (name[ll-1] == 'h' || name[ll-1] == 'c'))
	{
	FileType = CFILE;
	return;
	}


    if (ll > 1 && name[ll-2] == '.' && name[ll-1] == 's')
	{
	FileType = ASMFILE;
	return;
	}

    if ((ll > 3 && strcmp(&name[ll-4], ".rpc") == 0)
    	|| (ll > 4 && strcmp(&name[ll-5], ".rpc2") == 0))
	{
	FileType = RPCFILE;
	return;
	}

    FileType = UNKNOWN;
    }

